unit Document;


interface

uses
  System.Drawing,
  System.Drawing.Text,
  System.ComponentModel,
  System.Windows.Forms;

type
  TForm           = System.Windows.Forms.Form;
  TMainMenu       = System.Windows.Forms.MainMenu;
  TMenuItem       = System.Windows.Forms.MenuItem;
  TContainer      = System.ComponentModel.Container;
  TRichTextBox    = System.Windows.Forms.RichTextBox;
  TSize           = System.Drawing.Size;
  TFont           = System.Drawing.Font;
  TFontFamily     = System.Drawing.FontFamily;
  TOpenFileDialog = OpenFileDialog;

  TFontSizes   = (Small, Medium, Large);
const
  FontSizes: array[Low(TFontSizes)..High(TFontSizes)] of Single = (8, 12, 24);
  a: Integer = 5;

type
  TDocument = class(TForm)
  private
    FComponents: TContainer;
    FFontSize: Single;
    procedure InitComponents;
    procedure LoadDocumentClicked(Sender: TObject; Args: EventArgs);
    procedure FormatFontClicked(Sender: TObject; Args: EventArgs);
    procedure FormatSizeClicked(Sender: TObject; Args: EventArgs);
  published
    MainMenu: TMainMenu;
    RichTextBox: TRichTextBox;

    miFormatFontChecked: TMenuItem;
    miFormatSizeChecked: TMenuItem;

    miSmall: TMenuItem;
    miMedium: TMenuItem;
    miLarge: TMenuItem;

    miSansSerif: TMenuItem;
    miSerif: TMenuItem;
    miMonoSpace: TMenuItem;

    currentFontFamily: FontFamily;
    monoSpaceFontFamily: FontFamily;
    sansSerifFontFamily: FontFamily;
    serifFontFamily: FontFamily;    
  public
    constructor Create(Caption: string);
  end;

implementation

{ TDocument }


const
  b: Integer = 5;

constructor TDocument.Create(Caption: string);
var
  miFile, miFormat, miFontFace, miFontSize: TMenuItem;
begin
  inherited Create;
  InitComponents;

  Text:= Caption;

  // Initialize Fonts - use generic fonts to avoid problems across
  // different versions of the OS
  monoSpaceFontFamily:= TFontFamily.Create(GenericFontFamilies.Monospace);
  sansSerifFontFamily:= TFontFamily.Create(GenericFontFamilies.SansSerif);
  serifFontFamily:= TFontFamily.Create(GenericFontFamilies.Serif);
  currentFontFamily:= sansSerifFontFamily;

  RichTextBox.Font:= TFont.Create(currentFontFamily, FFontSize);
  RichTextBox.Text:= Caption;

  //Add File Menu
  miFile:= mainMenu.MenuItems.Add('&File');
  miFile.MergeType:= MenuMerge.MergeItems;
  miFile.MergeOrder:= 0;

  miFile.MenuItems.Add(TMenuItem.Create('&Load Document (' + Caption + ')', nil, Shortcut.CtrlL));
  miFile.MenuItems[0].Add_Click(LoadDocumentClicked);
  miFile.MenuItems[0].MergeOrder:= 105;

  //Add Formatting Menu
  miFormat:= TMenuItem.Create('F&ormat (' + Caption + ')', nil, Shortcut.CtrlO);
  miFormat.MergeType:= MenuMerge.Add;
  miFormat.MergeOrder:= 5;
  mainMenu.MenuItems.Add(miFormat);

  //Font Face sub-menu
  miSansSerif:= TMenuItem.Create;
  miSansSerif.Text:= '&1.' + sansSerifFontFamily.Name;
  miSansSerif.Add_Click(FormatFontClicked);

  miSerif:= TMenuItem.Create;
  miSerif.Text:='&2.' + serifFontFamily.Name;
  miSerif.Add_Click(FormatFontClicked);

  miMonoSpace:= TMenuItem.Create;
  miMonoSpace.Text:= '&3.' + monoSpaceFontFamily.Name;
  miMonoSpace.Add_Click(FormatFontClicked);

  miSansSerif.Checked:= True ;
  miFormatFontChecked:= miSansSerif ;
  miSansSerif.DefaultItem:= True ;

  miFontFace:= TMenuItem.Create;
  miFontFace.Text:= 'Font &Face';
  miFontFace.MenuItems.Add(miSansSerif);
  miFontFace.MenuItems.Add(miSerif);
  miFontFace.MenuItems.Add(miMonoSpace);

  miFormat.MenuItems.Add(miFontFace);

  //Font Size sub-menu
  miSmall:= TMenuItem.Create;
  miSmall.Text:= '&Small';
  miSmall.Add_Click(FormatSizeClicked);

  miMedium:= TMenuItem.Create;
  miMedium.Text:= '&Medium';
  miMedium.Add_Click(FormatSizeClicked);

  miLarge:= TMenuItem.Create;
  miLarge.Text:= '&Large';
  miLarge.Add_Click(FormatSizeClicked);

  miMedium.Checked:= True;
  miMedium.DefaultItem:= True;
  miFormatSizeChecked:= miMedium;

  miFontSize:= TMenuItem.Create;
  miFontSize.Text:= 'Font &Size';
  miFontSize.MenuItems.Add(miSmall);
  miFontSize.MenuItems.Add(miMedium);
  miFontSize.MenuItems.Add(miLarge);
  miFormat.MenuItems.Add(miFontSize);
end;

procedure TDocument.LoadDocumentClicked(Sender: TObject; Args: EventArgs);
var
  OpenDlg: TOpenFileDialog;
begin
  // Initialize the OpenFileDialog to look for RTF files.
  OpenDlg:= TOpenFileDialog.Create;
  OpenDlg.DefaultExt:= '*.txt';
  OpenDlg.Filter:= 'Text Files|*.txt';

  // Determine whether the user selected a file from the OpenFileDialog.
  if ( OpenDlg.ShowDialog = System.Windows.Forms.DialogResult.OK ) and ( OpenDlg.FileName <> '' )  then
    RichTextBox.LoadFile(OpenDlg.FileName, RichTextBoxStreamType.PlainText);
end;

procedure TDocument.FormatFontClicked(Sender: TObject; Args: EventArgs);
var
  miClicked: TMenuItem;
begin
  miClicked:= TMenuItem(Sender);
  miClicked.Checked:= True;
  miFormatFontChecked.Checked:= False;
  miFormatFontChecked:= miClicked ;

  if (miClicked = miSansSerif) then
    currentFontFamily:= sansSerifFontFamily
  else if (miClicked = miSerif) then
    currentFontFamily:= serifFontFamily
  else
    currentFontFamily:= monoSpaceFontFamily;

  RichTextBox.Font:= TFont.Create(currentFontFamily, FFontSize);
end;

procedure TDocument.FormatSizeClicked(Sender: TObject; Args: EventArgs);
var
  miClicked: TMenuItem;
begin
  miClicked:= TMenuItem(Sender);
  miClicked.Checked:= True;
  miFormatSizeChecked.Checked:= False;
  miFormatSizeChecked:= miClicked;

  if (miClicked = miSmall) then
    FFontSize:= FontSizes[Small]
  else if (miClicked = miLarge) then
    FFontSize:= FontSizes[Large]
  else
    FFontSize:= FontSizes[Medium];

  RichTextBox.Font:= TFont.Create(currentFontFamily, FFontSize);
end;

procedure TDocument.InitComponents;
begin
  SuspendLayout;

  FFontSize:= FontSizes[Medium];
  FComponents:= TContainer.Create;
  MainMenu:= TMainMenu.Create;
  RichTextBox:= TRichTextBox.Create;
  RichTextBox.Size:= TSize.Create(292, 273);
  RichTextBox.TabIndex:= 0;
  RichTextBox.Dock:= DockStyle.Fill;
  AutoScaleBaseSize:= TSize.Create(5, 13);
  ClientSize:= TSize.Create(392, 117);
  Menu:= MainMenu;
  Controls.Add(RichTextBox);

  ResumeLayout(False);
end;

end.
